package ModelFeature;
use strict;
use base 'Bio::SeqFeature::Generic';

# Specify labels and dependency order of valid feature types.
our $type_labels = 
{
  "gene"            => "gene",
  "transcript"      => "transcript",
  "mRNA"            => "transcript",
  "exon"            => "exon",
  "UTR"             => "UTR",
  "five_prime_UTR"  => "UTR",
  "three_prime_UTR" => "UTR",
  "CDS"             => "CDS",
};
our $type_order =
{
  "gene"            => 1,
  "transcript"      => 2,
  "mRNA"            => 2,
  "exon"            => 3,
  "UTR"             => 4,
  "five_prime_UTR"  => 4,
  "three_prime_UTR" => 4,
  "CDS"             => 5,
};

# Initialize from scratch
sub new
{
  my($class, @arguments) = @_;
  my $self = Bio::SeqFeature::Generic->new(@arguments);
  return initialize_from_generic($class, $self);
}

# Initialize from an existing Bio::SeqFeature::Generic object
sub initialize_from_generic
{
  my($class, $generic) = @_;
  return if(not defined($generic));
  if($generic->has_tag("ID"))
  {
    my @ids = $generic->get_tag_values("ID");
    my($id) = @ids;
    if((my $num_ids = scalar(@ids)) > 1)
    {
      my $id_string = sprintf("'%s'", join("', '", @ids));
      printf(STDERR "[ModelFeature] Warning: feature has %d ID attributes (%s), only keeping '%s'\n", $num_ids, $id_string, $id);
    }
    $generic->{_id} = $id;
  }
  bless($generic, $class);
  return $generic;
}

# Grab this feature's ID
sub id
{
  my($self) = @_;
  return $self->{_id};
}

# Grab this feature's type
sub type
{
  my($self) = @_;
  return $type_labels->{$self->primary_tag()};
}

# Is this feature valid?
sub valid
{
  my($self, $verbose) = @_;
  my $pt = $self->primary_tag();
  my $type = $type_labels->{$pt};

  my $valid_type = defined($type);
  printf(STDERR "[ModelFeature] Invalid: unrelated feature type '%s'\n", $pt) if(not $valid_type and $verbose);

  my $has_id = defined($self->{_id});
  my $need_id = ($type eq "gene" or $type eq "transcript");
  printf(STDERR "[ModelFeature] Invalid: feature of type '%s' must have ID\n", $type) if($need_id and not $has_id and $verbose);

  my $has_parent = $self->has_tag("Parent");
  my $need_parent = ($type eq "transcript" or $type eq "exon" or $type eq "CDS" or $type eq "UTR");
  printf(STDERR "[ModelFeature] Invalid: feature of type '%s' must have Parent\n", $type) if($need_parent and not $has_parent and $verbose);

  return ($valid_type and ($has_id or not $need_id) and ($has_parent or not $need_parent));
}

1;
