package ca.gc.phac.aspc.nml;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.biojava.bio.program.abi.ABIFParser;
import org.biojava.utils.io.CachingInputStream;

/**
 * A class for retrieving data from an ABIF formatted data file. The values are
 * calculated as per the ABI document titled
 * "ABIF File Format Specification and Sample File Schema". This document can be
 * found at: <a href="http://www.appliedbiosystems.com/support/software_community/ABIF_File_Format.pdf"
 * > http://www.appliedbiosystems.com/support/software_community/
 * ABIF_File_Format.pdf</a>. The Tibbets paper referred to in {@link ABIFParser}
 * is almost 15 years old at this point and the ABI spec document is less than
 * 5.
 * 
 * The accessor methods that are presented in this class are modeled after
 * Bio::Trace::ABIF by Nicola Vitacolonna.
 * 
 * @author Franklin Bristow (franklin_bristow@phac-aspc.gc.ca)
 * 
 */
public class ExtendedABIFParser extends ABIFParser {
	/** the list of possible bases in an ABI formatted file */
	private static final List<Character> BASES = Arrays.asList('A', 'C', 'G',
			'T');

	/**
	 * Creates a new ExtendedABIFParser for the specified
	 * {@link ABIFParser.DataAccess} object. If you need to read something from
	 * other than a file or a stream, you'll have to implement an
	 * {@link ABIFParser.DataAccess} class wrapping your source then pass an
	 * instance to this constructor.
	 * 
	 * @param dataAccess
	 * @throws IOException
	 */
	public ExtendedABIFParser(DataAccess dataAccess) throws IOException {
		super(dataAccess);
	}

	/**
	 * Creates a new ExtendedABIFParser for a file.
	 * 
	 * @param file
	 * @throws IOException
	 */
	public ExtendedABIFParser(File file) throws IOException {
		super(file);
	}

	/**
	 * Creates a new ExtendedABIFParser for an input stream. Note that the
	 * stream will be wrapped in a {@link CachingInputStream} if it isn't one
	 * already. If it is, it will be seeked to 0.
	 * 
	 * @param inputStream
	 * @throws IOException
	 */
	public ExtendedABIFParser(InputStream inputStream) throws IOException {
		super(inputStream);
	}

	/**
	 * The sequencing analysis program determines the clear range of the
	 * sequence by trimming bases from the 5' to 3' ends until fewer than 4
	 * bases out of 20 have a quality value less than 20. You can change these
	 * parameters by explicitly passing arguments to this method (the defaults
	 * are: windowWidth = 20, badBasesThreshold = 4, qualityThreshold = 20).
	 * Note that Sequencing Analysis counts the bases starting from one, so you
	 * have to add one to the return values to get consistent results.
	 * 
	 * @param windowWidth
	 *            the window width to use
	 * @param badBasesThreshold
	 *            the bad bases threshold to use
	 * @param qualityThreshold
	 *            the quality threshold to use
	 * 
	 * @return the clear range
	 */
	public Short[] getClearRange(Short windowWidth, Short badBasesThreshold,
			Short qualityThreshold) {
		if (windowWidth == null) {
			windowWidth = 20;
		}
		if (badBasesThreshold == null) {
			badBasesThreshold = 4;
		}
		if (qualityThreshold == null) {
			qualityThreshold = 20;
		}

		return new Short[] {
				getClearRangeStart(windowWidth, badBasesThreshold,
						qualityThreshold),
				getClearRangeStop(windowWidth, badBasesThreshold,
						qualityThreshold) };
	}

	/**
	 * Get the clear range start position.
	 * 
	 * @param windowWidth
	 *            the window width to use
	 * @param badBasesThreshold
	 *            the bad bases threshold to use
	 * @param qualityThreshold
	 *            the quality threshold to use
	 * @return the clear range start position
	 * @see ExtendedABIFParser#getClearRange(Short, Short, Short)
	 */
	public Short getClearRangeStart(Short windowWidth, Short badBasesThreshold,
			Short qualityThreshold) {

		if (windowWidth == null) {
			windowWidth = 20;
		}

		if (badBasesThreshold == null) {
			badBasesThreshold = 4;
		}

		if (qualityThreshold == null) {
			qualityThreshold = 20;
		}

		Short clearRangeStart = -1;
		Short[] qualityValues = getQualityValues();

		if (qualityValues.length >= windowWidth) {
			Short badBases = 0;
			Short i = 0;
			for (; i < windowWidth; i++) {
				if (qualityValues[i] < qualityThreshold) {
					badBases++;
				}
			}

			while (badBases >= badBasesThreshold && i < qualityValues.length) {
				if (qualityValues[i - windowWidth] < qualityThreshold) {
					badBases--;
				}

				if (qualityValues[i] < qualityThreshold) {
					badBases++;
				}

				i++;
			}

			if (badBases < badBasesThreshold) {
				clearRangeStart = Integer.valueOf(i - windowWidth).shortValue();
			}
		}

		return clearRangeStart;
	}

	/**
	 * Get the clear range stop position.
	 * 
	 * @param windowWidth
	 *            the window width to use
	 * @param badBasesThreshold
	 *            the bad bases threshold to use
	 * @param qualityThreshold
	 *            the quality threshold to use
	 * @return the clear range stop position
	 * @see ExtendedABIFParser#getClearRange(Short, Short, Short)
	 */
	public Short getClearRangeStop(Short windowWidth, Short badBasesThreshold,
			Short qualityThreshold) {

		if (windowWidth == null) {
			windowWidth = 20;
		}

		if (badBasesThreshold == null) {
			badBasesThreshold = 4;
		}

		if (qualityThreshold == null) {
			qualityThreshold = 20;
		}

		Short clearRangeStop = -1;
		Short[] qualityValues = getQualityValues();

		if (qualityValues.length >= windowWidth) {
			Short badBases = 0;
			Short i = Integer.valueOf(qualityValues.length - 1).shortValue();

			for (; i >= qualityValues.length - windowWidth; i--) {
				if (qualityValues[i] < qualityThreshold) {
					badBases++;
				}
			}

			while (badBases >= badBasesThreshold && i >= 0) {
				if (qualityValues[i + windowWidth] < qualityThreshold) {
					badBases--;
				}

				if (qualityValues[i] < qualityThreshold) {
					badBases++;
				}

				i--;
			}

			if (badBases < badBasesThreshold) {
				clearRangeStop = Integer.valueOf(i + windowWidth).shortValue();
			}
		}

		return clearRangeStop;
	}

	/**
	 * The sample score is the average quality value of the bases in the clear
	 * range of the sequence. This method returns 0 if the information needed to
	 * compute such value is missing or if the clear range is empty.
	 * 
	 * @param windowWidth
	 *            the window width
	 * @param badBasesThreshold
	 *            the bad bases threshold
	 * @param qualityThreshold
	 *            the quality threshold
	 * @return the sample score
	 * @see ExtendedABIFParser#getClearRange(Short, Short, Short)
	 */
	public Short getSampleScore(Short windowWidth, Short badBasesThreshold,
			Short qualityThreshold) {
		if (windowWidth == null) {
			windowWidth = 20;
		}

		if (badBasesThreshold == null) {
			badBasesThreshold = 4;
		}

		if (qualityThreshold == null) {
			qualityThreshold = 20;
		}

		Short[] qualityValues = getQualityValues();
		Short sampleScore = 0;

		if (qualityValues.length > 0) {
			Short[] clearRangeStartStop = getClearRange(windowWidth,
					badBasesThreshold, qualityThreshold);
			Short start = clearRangeStartStop[0];
			Short stop = clearRangeStartStop[1];

			if (start >= 0 && start <= stop) {

				Long sum = 0L;

				for (Short i = start; i <= stop; i++) {
					sum += qualityValues[i];
				}

				sampleScore = Long.valueOf(sum / (stop - start + 1))
						.shortValue();
			}
		}

		return sampleScore;
	}

	/**
	 * There are four channels in an ABIF file, numbered from 9 to 12. An
	 * optional 5th channel exists (205).
	 * 
	 * ABIF tags: DATA9, DATA10, DATA11, DATA12 (and sometimes DATA205)
	 * 
	 * @param channelNumber
	 *            the channel that we should be returning
	 * @return the analyzed data for the specified channel
	 * @throws IndexOutOfBoundsException
	 *             if the specified channel number is not within the range of
	 *             the applicable channel numbers
	 */
	public Short[] getAnalyzedDataForChannel(Integer channelNumber)
			throws IndexOutOfBoundsException {
		if (channelNumber < 9 || (channelNumber > 12 && channelNumber != 205)) {
			throw new IndexOutOfBoundsException(
					"Invalid channel number specified, must be between 9 and 12 or 205");
		}

		return charArrayAsShorts(parseAsciiArray("DATA", channelNumber));
	}

	/**
	 * Get the analysis protocol settings name.
	 * 
	 * ABIF tag: APrN1
	 * 
	 * @return the analysis protocol settings name
	 */
	public String getAnalysisProtocolSettingsName() {
		return parseCString("APrN", 1);
	}

	/**
	 * Get the analysis protocol settings version.
	 * 
	 * ABIF tag: APrV1
	 * 
	 * @return the analysis protocol settings version
	 */
	public String getAnalysisProtocolSettingsVersion() {
		return parseCString("APrV", 1);
	}

	/**
	 * Get the analysis protocol xml string.
	 * 
	 * ABIF tag: APrX1
	 * 
	 * @return the analysis protocol xml string
	 */
	public String getAnalysisProtocolXml() {
		return new String(parseAsciiArray("APrX", 1));
	}

	/**
	 * Get the analysis protocol xml schema version.
	 * 
	 * ABIF tag: APXV1
	 * 
	 * @return the analysis protocol xml schema version
	 */
	public String getAnalysisProtocolXmlSchemaVersion() {
		return parseCString("APXV", 1);
	}

	/**
	 * Get the analysis return code.
	 * 
	 * ABIF tag: ARTN1
	 * 
	 * @return the analysis return code
	 */
	public Long getAnalysisReturnCode() {
		return parseLong("ARTN", 1);
	}

	/**
	 * Get the average peak spacing used in last analysis.
	 * 
	 * ABIF tag: SPAC1
	 * 
	 * @return the average peak spacing used in last analysis
	 */
	public Float getAvgPeakSpacing() {
		return parseFloat("SPAC", 1);
	}

	/**
	 * Get the basecaller adaptive processing success flag.
	 * 
	 * ABIF tag: ASPF1
	 * 
	 * @return the basecaller adaptive processing success flag
	 */
	public Short getBasecallerAspf() {
		return parseInteger("ASPF", 1).shortValue();
	}

	/**
	 * Get a string with the basecalled BCP/DLL.
	 * 
	 * ABIF tag: SPAC2
	 * 
	 * @return a string with the basecalled BCP/DLL
	 */
	public String getBasecallerBcpDll() {
		return parsePString("SPAC", 2);
	}

	/**
	 * Get the basecaller version (e.g., 'KB 1.3.0').
	 * 
	 * ABIF tag: SVER2
	 * 
	 * @return the basecaller version (e.g., 'KB 1.3.0')
	 */
	public String getBasecallerVersion() {
		return parsePString("SVER", 2);
	}

	/**
	 * Get the timestamp for the last successful basecalling analysis.
	 * 
	 * ABIF tag: BCTS1
	 * 
	 * @return the timestamp for the last successful basecalling analysis
	 */
	public String getBasecallingAnalysisTimestamp() {
		return parsePString("BCTS", 1);
	}

	/**
	 * Get the list of base locations.
	 * 
	 * ABIF tag: PLOC2
	 * 
	 * @return the list of base locations
	 */
	public char[] getBaseLocations() {
		return parseAsciiArray("PLOC", 2);
	}

	/**
	 * Get the list of edited base locations.
	 * 
	 * ABIF tag: PLOC1
	 * 
	 * @return the list of edited base locations
	 */
	public char[] getBaseLocationsEdited() {
		return parseAsciiArray("PLOC", 1);
	}

	/**
	 * Get the array of characters sorted by channel number. For example, if the
	 * list is ('G', 'A', 'T', 'C') then G is channel 1, A is channel 2, and so
	 * on.
	 * 
	 * ABIF tag: FWO_1
	 * 
	 * @return the array of characters sorted by channel number
	 */
	public char[] getBaseOrder() {
		return parseAsciiArray("FWO_", 1);
	}

	/**
	 * Get the spacing of the bases.
	 * 
	 * ABIF tag: SPAC3
	 * 
	 * @return the spacing of the bases
	 */
	public Float getBaseSpacing() {
		return parseFloat("SPAC", 3);
	}

	/**
	 * Get the buffer tray heater temperature in &#176;C.
	 * 
	 * ABIF tag: BufT1
	 * 
	 * @return the buffer tray heater temperature in &#176;C
	 */
	public Short[] getBufferTrayTemperature() {
		return charArrayAsShorts(parseAsciiArray("BufT", 1));
	}

	/**
	 * Get the lane or capillary number for this sample.
	 * 
	 * ABIF tag: LANE1
	 * 
	 * @return the lane or capillary number for this sample
	 */
	public Short getCapillaryNumber() {
		return parseInteger("LANE", 1).shortValue();
	}

	/**
	 * Get the channel number corresponding to the provided base.
	 * 
	 * @param base
	 *            the base to find the channel for
	 * @return the channel number corresponding to the provided base
	 * @throws IndexOutOfBoundsException
	 *             if base is not one of 'A', 'C', 'G', or 'T'
	 */
	public Short getChannel(char base) throws IndexOutOfBoundsException {
		if (!BASES.contains(base)) {
			throw new IndexOutOfBoundsException(
					"Invalid base provided for channel");
		}

		char[] baseOrder = getBaseOrder();
		Short basePosition = null;
		for (Short i = 0; i < baseOrder.length; i++) {
			if (baseOrder[i] == base) {
				basePosition = i;
				break;
			}
		}
		return basePosition;
	}

	/**
	 * Get the primer or terminator chemistry.
	 * 
	 * ABIF tag: phCH1
	 * 
	 * @return the primer or terminator chemistry
	 */
	public String getChem() {
		return parsePString("phCH", 1);
	}

	/**
	 * Get the comment regarding the sample. This is an optional data item. In
	 * some files there is more than one comment. The optional argument is used
	 * to specify the number of the comment. If the comment with the specified
	 * number does not exist, null is returned.
	 * 
	 * ABIF tag: CMNT1 ... CMNT 'N'
	 * 
	 * @param n
	 *            The desired comment number to be retrieved. Defaults to 1 if
	 *            null is passed.
	 * @return the comment regarding the sample with the specified index
	 */
	public String getComment(Integer n) {
		return parsePString("CMNT", n);
	}

	/**
	 * Get the comment title.
	 * 
	 * ABIF tag: CTTL1
	 * 
	 * @return the comment title
	 */
	public String getCommentTitle() {
		return parsePString("CTTL", 1);
	}

	/**
	 * Get the container identifier, a.k.a. plate barcode.
	 * 
	 * ABIF tag: CTID1
	 * 
	 * @return the container identifier
	 */
	public String getContainerIdentifier() {
		return parseCString("CTID", 1);
	}

	/**
	 * Get the container name. Usually, this is identical to the container
	 * identifier.
	 * 
	 * ABIF tag: CTMN1
	 * 
	 * @return the container name
	 */
	public String getContainerName() {
		return parseCString("CTNM", 1);
	}

	/**
	 * Get the container's owner.
	 * 
	 * ABIF tag: CTow1
	 * 
	 * @return the container's owner
	 */
	public String getContainerOwner() {
		return parseCString("CTow", 1);
	}

	/**
	 * Get the current, measured in milliamps.
	 * 
	 * ABIF tag: DATA6
	 * 
	 * @return the current, measured in milliamps
	 */
	public Short[] getCurrent() {
		return charArrayAsShorts(parseAsciiArray("DATA", 6));
	}

	/**
	 * Get the data collection module file.
	 * 
	 * ABIF tag: MODF1
	 * 
	 * @return the data collection module file
	 */
	public String getDataCollectionModuleFile() {
		return parsePString("MODF", 1);
	}

	/**
	 * Get the data collection software version.
	 * 
	 * ABIF tag: SVER1
	 * 
	 * @return the data collection software version
	 */
	public String getDataCollectionSoftwareVersion() {
		return parsePString("SVER", 1);
	}

	/**
	 * Get the data collection firmware version.
	 * 
	 * ABIF tag: SVER3
	 * 
	 * @return the data collection firmware version
	 */
	public String getDataCollectionFirmwareVersion() {
		return parsePString("SVER", 3);
	}

	/**
	 * Get the data collection start date.
	 * 
	 * ABIF tag: RUND3
	 * 
	 * @return the data collection start date
	 */
	public Date getDataCollectionStartDate() {
		return parseDate("RUND", 3);
	}

	/**
	 * Get the data collection start time. Formatted as HH:mm:ss.SS using
	 * {@link SimpleDateFormat} and thus suitable for parsing.
	 * 
	 * ABIF tag: RUNT3
	 * 
	 * @return the data collection start time
	 */
	public String getDataCollectionStartTime() {
		return parseTime("RUNT", 3);
	}

	/**
	 * Convenience method for retrieving the values of ABIF tags RUND3 and RUNT3
	 * as a single date object.
	 * 
	 * @return the data collection start date and time as a single date object
	 */
	public Date getDataCollectionStartDateTime() {
		return addTimeToDate(getDataCollectionStartDate(),
				getDataCollectionStartTime(), "HH:mm:ss.SS");
	}

	/**
	 * Get the data collection stop date.
	 * 
	 * ABIF tag: RUND4
	 * 
	 * @return the data collection stop date
	 */
	public Date getDataCollectionStopDate() {
		return parseDate("RUND", 4);
	}

	/**
	 * Get the data collection stop time. Formatted as HH:mm:ss.SS using
	 * {@link SimpleDateFormat} and thus suitable for parsing.
	 * 
	 * ABIF tag: RUNT4
	 * 
	 * @return the data collection stop time
	 */
	public String getDataCollectionStopTime() {
		return parseTime("RUNT", 4);
	}

	/**
	 * Convenience method for retrieving the values of ABIF tags RUND4 and RUNT4
	 * as a single date object.
	 * 
	 * @return the data collection start date and time as a single date object
	 */
	public Date getDataCollectionStopDateTime() {
		return addTimeToDate(getDataCollectionStopDate(),
				getDataCollectionStopTime(), "HH:mm:ss.SS");
	}

	/**
	 * Get the detector cell heater temperature in &#176;C.
	 * 
	 * ABIF tag: DCHT1
	 * 
	 * @return the detector cell heater temperature in &#176;C
	 */
	public Short getDetectorHeaterTemperature() {
		return parseInteger("DCHT", 1).shortValue();
	}

	/**
	 * Get the downsampling factor.
	 * 
	 * ABIF tag: DSam1
	 * 
	 * @return the downsampling factor
	 */
	public Short getDownsamplingFactor() {
		return parseInteger("DSam", 1).shortValue();
	}

	/**
	 * Get the name of dye number at dyeNum. dyeNum must be between 1 and 5.
	 * DyeN5 is an optional field and may not be populated.
	 * 
	 * ABIF tag: DyeN1, DyeN2, DyeN3, DyeN4 (and sometimes DyeN5)
	 * 
	 * @param dyeNum
	 *            the number of the desired dye number
	 * @return the dye name for the specified dye number
	 * @throws IndexOutOfBoundsException
	 *             if dyeNum is outside the range 1-5
	 */
	public String getDyeName(Short dyeNum) throws IndexOutOfBoundsException {
		if (dyeNum < 1 || dyeNum > 5) {
			throw new IndexOutOfBoundsException(
					"Invalid dyeNum, dyeNum must be between 1 and 5");
		}

		return parsePString("DyeN", dyeNum.intValue());
	}

	/**
	 * Get the dye set name.
	 * 
	 * ABIF tag: DySN1
	 * 
	 * @return the dye set name
	 */
	public String getDyeSetName() {
		return parsePString("DySN", 1);
	}

	/**
	 * Get the dye significance for dyeNum. dyeNum must be between 1 and 5.
	 * DyeB5 is an optional field and may not be populated. Returned value of
	 * 'S' implies standard, null implies sample.
	 * 
	 * ABIF tag: DyeB1, DyeB2, DyeB3, DyeB4 (and sometimes DyeB5)
	 * 
	 * @param dyeNum
	 *            the number of the desired dye number
	 * @return the significance value of the specified dye number
	 * @throws IndexOutOfBoundsException
	 *             if dyeNum is outside the range 1-5
	 */
	public Character getDyeSignificance(Short dyeNum)
			throws IndexOutOfBoundsException {
		if (dyeNum < 1 || dyeNum > 5) {
			throw new IndexOutOfBoundsException(
					"Invalid dyeNum, dyeNum must be between 1 and 5");
		}

		return parseAsciiArray("DyeB", dyeNum.intValue())[0];
	}

	/**
	 * Get the dye type.
	 * 
	 * ABIF tag: phDY1
	 * 
	 * @return the dye type
	 */
	public String getDyeType() {
		return parsePString("phDY", 1);
	}

	/**
	 * Get the dye wavelength for dyeNum. dyeNum must be between 1 and 5. DyeW5
	 * is an optional field and may not be populated.
	 * 
	 * ABIF tag: DyeW1, DyeW2, DyeW3, DyeW4 (and sometimes DyeW5)
	 * 
	 * @param dyeNum
	 *            the number of the desired dye number
	 * @return the wavelength value of the specified dye number
	 * @throws IndexOutOfBoundsException
	 *             if dyeNum is outside the range 1-5
	 */
	public Short getDyeWavelength(Short dyeNum)
			throws IndexOutOfBoundsException {
		if (dyeNum < 1 || dyeNum > 5) {
			throw new IndexOutOfBoundsException(
					"Invalid dyeNum, dyeNum must be between 1 and 5");
		}

		return parseInteger("DyeW", dyeNum.intValue()).shortValue();
	}

	/**
	 * Get the list of edited quality values. I am not yet convinced that this
	 * should be in a char array....
	 * 
	 * ABIF tag: PCON1
	 * 
	 * @return the list of edited quality values
	 */
	public char[] getEditedQualityValues() {
		return parseAsciiArray("PCON", 1);
	}

	/**
	 * Get the string of the edited basecalled sequence.
	 * 
	 * ABIF tag: PBAS1
	 * 
	 * @return the string of the edited basecalled sequence
	 */
	public char[] getEditedSequence() {
		return parseAsciiArray("PBAS", 1);
	}

	/**
	 * Get the electrophoresis voltage setting in volts.
	 * 
	 * ABIF tag: EPVt1
	 * 
	 * @return the electrophoresis voltage setting in volts
	 */
	public Long getElectrophoresisVoltage() {
		return parseLong("EPVt", 1);
	}

	/**
	 * Get the gel type description.
	 * 
	 * ABIF tag: GTyp1
	 * 
	 * @return the gel type description
	 */
	public String getGelType() {
		return parsePString("GTyp", 1);
	}

	/**
	 * Get the GeneMapper(R) software analysis method name.
	 * 
	 * ABIF tag: ANME1
	 * 
	 * @return the GeneMapper(R) software analysis method name
	 */
	public String getGeneMapperAnalysisMethod() {
		return parseCString("ANME", 1);
	}

	/**
	 * Get the GeneMapper(R) software panel name.
	 * 
	 * ABIF tag: PANL1
	 * 
	 * @return the GeneMapper(R) software panel name
	 */
	public String getGeneMapperPanelName() {
		return parseCString("PANL", 1);
	}

	/**
	 * Get the GeneMapper(R) software Sample Type.
	 * 
	 * ABIF tag: STYP1
	 * 
	 * @return the GeneMapper(R) software Sample Type
	 */
	public String getGeneMapperSampleType() {
		return parseCString("STYP", 1);
	}

	/**
	 * Get the sample name for GeneScan(R) sample files.
	 * 
	 * ABIF tag: SpNm1
	 * 
	 * @return the sample name for GeneScan(R) sample files
	 */
	public String getGeneScanSampleName() {
		return parsePString("SpNm", 1);
	}

	/**
	 * Get the injection time in seconds.
	 * 
	 * ABIF tag: InSc1
	 * 
	 * @return the injection time in seconds
	 */
	public Long getInjectionTime() {
		return parseLong("InSc", 1);
	}

	/**
	 * Get the injection voltage in volts.
	 * 
	 * ABIF tag: InVt1
	 * 
	 * @return the injection voltage in volts
	 */
	public Long getInjectionVoltage() {
		return parseLong("InVt", 1);
	}

	/**
	 * Get the instrument class.
	 * 
	 * ABIF tag: HCFG1
	 * 
	 * @return the instrument class
	 */
	public String getInstrumentClass() {
		return parseCString("HCFG", 1);
	}

	/**
	 * Get the instrument family.
	 * 
	 * ABIF tag: HCFG2
	 * 
	 * @return the instrument family
	 */
	public String getInstrumentFamily() {
		return parseCString("HCFG", 2);
	}

	/**
	 * Get the instrument name and serial number.
	 * 
	 * ABIF tag: MCHN1
	 * 
	 * @return the instrument name and serial number
	 */
	public String getInstrumentNameAndSerialNumber() {
		return parsePString("MCHN", 1);
	}

	/**
	 * Get the instrument parameters.
	 * 
	 * ABIF tag: HCFG4
	 * 
	 * @return the instrument parameters
	 */
	public String getInstrumentParam() {
		return parseCString("HCFG", 4);
	}

	/**
	 * Is this machine a capillary machine? This is one of those crazy ones
	 * where biojava is parsing this field into an ascii array even though it is
	 * just supposed to be a single byte.
	 * 
	 * ABIF tag: CpEP1
	 * 
	 * @return whether or not this machine is a capillary machine
	 */
	public Boolean isCapillaryMachine() {
		return parseAsciiArray("CpEP", 1)[0] > 0;
	}

	/**
	 * Get the laser power setting in microwatts.
	 * 
	 * ABIF tag: LsrP1
	 * 
	 * @return the laser power setting in microwatts
	 */
	public Long getLaserPower() {
		return parseLong("LsrP", 1);
	}

	/**
	 * Get the length to the detector in cm.
	 * 
	 * ABIF tag: LNTD1
	 * 
	 * @return the length to the detector in cm
	 */
	public Short getLengthToDetector() {
		return parseInteger("LNTD", 1).shortValue();
	}

	/**
	 * Get the name of the mobility file.
	 * 
	 * ABIF tag: PDMF2
	 * 
	 * @return the name of the mobility file
	 */
	public String getMobilityFile() {
		return parsePString("PDMF", 2);
	}

	/**
	 * Get the name of the mobility file (orig).
	 * 
	 * ABIF tag: PDMF1
	 * 
	 * @return the name of the mobility file (orig)
	 */
	public String getMobilityFileOrig() {
		return parsePString("PDMF", 1);
	}

	/**
	 * Get the model number.
	 * 
	 * ABIF tag: MODL1
	 * 
	 * @return the model number
	 */
	public String getModelNumber() {
		return new String(parseAsciiArray("MODL", 1));
	}

	/**
	 * Get the number of capillaries.
	 * 
	 * ABIF tag: NLNE1
	 * 
	 * @return the number of capillaries
	 */
	public Short getNumCapillaries() {
		return parseInteger("NLNE", 1).shortValue();
	}

	/**
	 * Get the number of dyes.
	 * 
	 * ABIF tag: Dye#1
	 * 
	 * @return the number of dyes
	 */
	public Short getNumDyes() {
		return parseInteger("Dye#", 1).shortValue();
	}

	/**
	 * Get the number of scans.
	 * 
	 * ABIF tag: SCAN1
	 * 
	 * @return the number of scans
	 */
	public Long getNumScans() {
		return parseLong("SCAN", 1);
	}

	/**
	 * Get the official instrument name.
	 * 
	 * ABIF tag: HCFG3
	 * 
	 * @return the official instrument name
	 */
	public String getOfficialInstrumentName() {
		return parseCString("HCFG", 3);
	}

	/**
	 * Get the range of offscale peaks. This data item's type is a user defined
	 * data structure. As such, it is returned as a list of bytes that must be
	 * interpreted by the caller. This is an optional data item.
	 * 
	 * ABIF tag: OffS1 ... OffS 'N'
	 * 
	 * @param peakNum
	 *            the index of offscale peaks to return
	 * @return the range of offscale peaks
	 */
	public Byte[] getOffscalePeaks(Integer peakNum) {
		return charArrayAsBytes(parseAsciiArray("OffS", peakNum));
	}

	/**
	 * Get the list of scans that are marked as off scale in Collection. This is
	 * an optional data item.
	 * 
	 * ABIF tag: OfSc1
	 * 
	 * @return the list of offscale scans
	 */
	public Long[] getOffscaleScans() {
		return charArrayAsLongs(parseAsciiArray("OfSc", 1));
	}

	/**
	 * Get the location of peak 1.
	 * 
	 * ABIF tag: B1Pt2
	 * 
	 * @return the location of peak 1
	 */
	public Short getPeak1Location() {
		return parseInteger("B1Pt", 2).shortValue();
	}

	/**
	 * Get the location of peak 1 (orig).
	 * 
	 * ABIF tag: B1Pt1
	 * 
	 * @return the location of peak 1 (orig)
	 */
	public Short getPeak1LocationOrig() {
		return parseInteger("B1Pt", 1).shortValue();
	}

	/**
	 * Get the peak area ratio.
	 * 
	 * ABIF tag: phAR1
	 * 
	 * @return the peak area ratio
	 */
	public Float getPeakAreaRatio() {
		return parseFloat("phAR", 1);
	}

	/**
	 * Get the pixel bin size.
	 * 
	 * ABIF tag: PXLB1
	 * 
	 * @return the pixel bin size
	 */
	public Long getPixelBinSize() {
		return parseLong("PXLB", 1);
	}

	/**
	 * Get the pixels averaged per lane.
	 * 
	 * ABIF tag: NAVG1
	 * 
	 * @return the pixels averaged per lane
	 */
	public Short getPixelsPerLane() {
		return parseInteger("NAVG", 1).shortValue();
	}

	/**
	 * Get the plate type.
	 * 
	 * ABIF tag: PTYP1
	 * 
	 * @return the plate type
	 */
	public String getPlateType() {
		return parseCString("PTYP", 1);
	}

	/**
	 * Get the number of sample positions in the container.
	 * 
	 * ABIF tag: PSZE1
	 * 
	 * @return the number of sample positions in the container
	 */
	public Long getPlateSize() {
		return parseLong("PSZE", 1);
	}

	/**
	 * Get the polymer lot expiration date.
	 * 
	 * ABIF tag: SMED1
	 * 
	 * @return the polymer lot expiration date
	 */
	public String getPolymerExpirationDate() {
		return parsePString("SMED", 1);
	}

	/**
	 * Get the polymer lot number.
	 * 
	 * ABIF tag: SMLt1
	 * 
	 * @return the polymer lot number
	 */
	public String getPolymerLotNumber() {
		return parsePString("SMLt", 1);
	}

	/**
	 * Get the power, measured in milliwatts.
	 * 
	 * ABIF tag: DATA7
	 * 
	 * @return the power, measured in milliwats
	 */
	public Short[] getPower() {
		return charArrayAsShorts(parseAsciiArray("DATA", 7));
	}

	/**
	 * Get the quality levels.
	 * 
	 * ABIF tag: phQL1
	 * 
	 * @return the quality levels
	 */
	public Short getQualityLevels() {
		return parseInteger("phQL", 1).shortValue();
	}

	/**
	 * Get the list of quality values.
	 * 
	 * ABIF tag: PCON2
	 * 
	 * @return the list of quality values
	 */
	public Short[] getQualityValues() {
		return charArrayAsShorts(parseAsciiArray("PCON", 2));
	}

	/**
	 * Get the raw data for the specified channel number. There are four
	 * channels in an ABIF file, numbered 1 to 4. An optional channel number 105
	 * exists in some files.
	 * 
	 * ABIF tag: DATA1, DATA2, DATA3, DATA4 (and sometimes DATA105)
	 * 
	 * @param channelNumber
	 *            the channel we should get the raw data for
	 * @return the raw data for the specified channel
	 * @throws IndexOutOfBoundsException
	 *             if the channel number is not between 1 and 4 or 105
	 */
	public Short[] getRawDataForChannel(Integer channelNumber)
			throws IndexOutOfBoundsException {
		if (channelNumber < 1 || (channelNumber > 4 && channelNumber != 105)) {
			throw new IndexOutOfBoundsException(
					"Invalid channel number, channel number must be between 1 and 4 or 105");
		}

		return charArrayAsShorts(parseAsciiArray("DATA", channelNumber));
	}

	/**
	 * Get the rescaling divisor for colour data.
	 * 
	 * ABIF tag: Scal1
	 * 
	 * @return the rescaling divisor for colour data
	 */
	public Float getRescaling() {
		return parseFloat("Scal", 1);
	}

	/**
	 * Get the results group name.
	 * 
	 * ABIF tag: RGNm1
	 * 
	 * @return the results group name
	 */
	public String getResultsGroup() {
		return parseCString("RGNm", 1);
	}

	/**
	 * Get the results group comment.
	 * 
	 * ABIF tag: RGCm1
	 * 
	 * @return the results group comment
	 */
	public String getResultsGroupComment() {
		return parseCString("RGCw", 1);
	}

	/**
	 * Get the results group owner.
	 * 
	 * ABIF tag: RGOw1
	 * 
	 * @return the results group owner
	 */
	public String getResultsGroupOwner() {
		return parseCString("RGOw", 1);
	}

	/**
	 * Get the reverse complement flag.
	 * 
	 * ABIF tag: RevC1
	 * 
	 * @return the reverse complement flag
	 */
	public Short getReverseComplementFlag() {
		return parseInteger("RevC", 1).shortValue();
	}

	/**
	 * Get the run module name.
	 * 
	 * ABIF tag: RMdN1
	 * 
	 * @return the run module name
	 */
	public String getRunModuleName() {
		return parseCString("RMdN", 1);
	}

	/**
	 * Get the run module version.
	 * 
	 * ABIF tag: RMdV1
	 * 
	 * @return the run module version
	 */
	public String getRunModuleVersion() {
		return parseCString("RMdV", 1);
	}

	/**
	 * Get the run module xml schema version.
	 * 
	 * ABIF tag: RMXV1
	 * 
	 * @return the run module xml schema version
	 */
	public String getRunModuleXmlSchemaVersion() {
		return parseCString("RMXV", 1);
	}

	/**
	 * Get the run module xml string.
	 * 
	 * ABIF tag: RMdX1
	 * 
	 * @return the run module xml string
	 */
	public char[] getRunModuleXmlString() {
		return parseAsciiArray("RMdX", 1);
	}

	/**
	 * Get the run name.
	 * 
	 * ABIF tag: RunN1
	 * 
	 * @return the run name
	 */
	public String getRunName() {
		return parseCString("RunN", 1);
	}

	/**
	 * Get the run protocol name.
	 * 
	 * ABIF tag: RPrN1
	 * 
	 * @return the run protocol name
	 */
	public String getRunProtocolName() {
		return parseCString("RPrN", 1);
	}

	/**
	 * Get the run protocol version.
	 * 
	 * ABIF tag: RPrV1
	 * 
	 * @return the run protocol version
	 */
	public String getRunProtocolVersion() {
		return parseCString("RPrV", 1);
	}

	/**
	 * Get the run start date.
	 * 
	 * ABIF tag: RUND1
	 * 
	 * @return the run start date
	 */
	public Date getRunStartDate() {
		return parseDate("RUND", 1);
	}

	/**
	 * Get the run start time.
	 * 
	 * ABIF tag: RUNT1
	 * 
	 * @return the run start time
	 */
	public String getRunStartTime() {
		return parseTime("RUNT", 1);
	}

	/**
	 * A convenience method for getting both RUND1 and RUNT1 in one value.
	 * 
	 * @return the run date and time as a single date value
	 */
	public Date getRunStartDateTime() {
		return addTimeToDate(getRunStartDate(), getRunStartTime(),
				"HH:mm:ss.SS");
	}

	/**
	 * Get the run stop date.
	 * 
	 * ABIF tag: RUND2
	 * 
	 * @return the run stop date
	 */
	public Date getRunStopDate() {
		return parseDate("RUND", 2);
	}

	/**
	 * Get the run stop time.
	 * 
	 * ABIF tag: RUNT2
	 * 
	 * @return the run stop time
	 */
	public String getRunStopTime() {
		return parseTime("RUNT", 2);
	}

	/**
	 * A convenience method for getting both RUND2 and RUNT2 in one values
	 * 
	 * @return the run date and time as a single value
	 */
	public Date getRunStopDateTime() {
		return addTimeToDate(getRunStopDate(), getRunStopTime(), "HH:mm:ss.SS");
	}

	/**
	 * Get the run temperature setting in &#176;C.
	 * 
	 * ABIF tag: Tmpr1
	 * 
	 * @return the run temperature setting in &176;C
	 */
	public Long getRunTemperature() {
		return parseLong("Tmpr", 1);
	}

	/**
	 * Get the sample file format version.
	 * 
	 * ABIF tag: SVER4
	 * 
	 * @return the sample file format version
	 */
	public String getSampleFileFormatVersion() {
		return parsePString("SVER", 4);
	}

	/**
	 * Get the sample name.
	 * 
	 * ABIF tag: SMPL1
	 * 
	 * @return the sample name
	 */
	public String getSampleName() {
		return parsePString("SMPL", 1);
	}

	/**
	 * Get the sample tracking id.
	 * 
	 * ABIF tag: LIMS1
	 * 
	 * @return the sample tracking id
	 */
	public String getSampleTrackingId() {
		return parsePString("LIMS", 1);
	}

	/**
	 * Get the scanning rate. This data item's type is a user defined data
	 * structure. As such, it is returned as a list of bytes that must be
	 * interpreted by the caller.
	 * 
	 * ABIF tag: Rate1
	 * 
	 * @return the scanning rate
	 */
	public Byte[] getScanningRate() {
		return charArrayAsBytes(parseAsciiArray("Rate", 1));
	}

	/**
	 * Get a list of data colour values for the locations listed by
	 * getScanNumberIndices().
	 * 
	 * ABIF tag: OvrV1 ... OvrV 'N'
	 * 
	 * @param index
	 *            the index to retrieve the colour data for
	 * @return the colour data for the specified index
	 */
	public Long[] getScanColourDataValues(Integer index) {
		return charArrayAsLongs(parseAsciiArray("OvrV", index));
	}

	/**
	 * Get the scan numbers of data points.
	 * 
	 * ABIF tag: Satd1
	 * 
	 * @return the scan number of data points
	 */
	public Long[] getScanNumbers() {
		return charArrayAsLongs(parseAsciiArray("Satd", 1));
	}

	/**
	 * Get a list of scan number indices for the specified index.
	 * 
	 * ABIF tag: OvrI1 ... OvrI 'N'
	 * 
	 * @param index
	 *            the index to retrieve scan number indices
	 * @return the scan number indices for the specified index.
	 */
	public Long[] getScanNumberIndices(Integer index) {
		return charArrayAsLongs(parseAsciiArray("OvrI", index));
	}

	/**
	 * Get the SeqScape(R) project name.
	 * 
	 * ABIF tag: PROJ4
	 * 
	 * @return the SeqScape(R) project name
	 */
	public String getSeqScapeProjectName() {
		return parseCString("PROJ", 4);
	}

	/**
	 * Get the SeqScape(R) project template name.
	 * 
	 * ABIF tag: PRJT1
	 * 
	 * @return the SeqScape(R) project template name
	 */
	public String getSeqScapeProjectTemplate() {
		return parseCString("PRJT", 1);
	}

	/**
	 * Get the SeqScape(R) specimen name.
	 * 
	 * ABIF tag: SPEC1
	 * 
	 * @return the SeqScape(R) specimen name
	 */
	public String getSeqScapeSpecimenName() {
		return parseCString("SPEC", 1);
	}

	/**
	 * Get the basecalled sequence.
	 * 
	 * ABIF tag: PBAS2
	 * 
	 * @return the basecalled sequence
	 */
	public char[] getSequence() {
		return parseAsciiArray("PBAS", 2);
	}

	/**
	 * Get the sequencing analysis parameters filename.
	 * 
	 * ABIF tag: APFN2
	 * 
	 * @return the sequencing analysis parameters filename.
	 */
	public String getSequencingAnalysisParamFilename() {
		return parsePString("APFN", 2);
	}

	/**
	 * Get the signal level for each dye.
	 * 
	 * ABIF tag: S/N%1
	 * 
	 * @return the signal level for each dye
	 */
	public Short[] getSignalLevel() {
		return parseShortArray("S/N%", 1);
	}

	/**
	 * Get the size standard file name.
	 * 
	 * ABIF tag: StdF1
	 * 
	 * @return the size standard file name.
	 */
	public String getSizeStandardFilename() {
		return parsePString("StdF", 1);
	}

	/**
	 * Get the SNP set name.
	 * 
	 * ABIF tag: SnpS1
	 * 
	 * @return the SNP set name
	 */
	public String getSnpSetName() {
		return parsePString("SnpS", 1);
	}

	/**
	 * Get the start collection event.
	 * 
	 * ABIF tag: EVNT3
	 * 
	 * @return the start collection event
	 */
	public String getStartCollectionEvent() {
		return parsePString("EVNT", 3);
	}

	/**
	 * Get the start point.
	 * 
	 * ABIF tag: ASPt2
	 * 
	 * @return the start point
	 */
	public Short getStartPoint() {
		return parseInteger("ASPt", 2).shortValue();
	}

	/**
	 * Get the start point (orig).
	 * 
	 * ABIF tag: ASPt1
	 * 
	 * @return the start point (orig)
	 */
	public Short getStartPointOrig() {
		return parseInteger("ASPt", 1).shortValue();
	}

	/**
	 * Get the start run event.
	 * 
	 * ABIF tag: EVNT1
	 * 
	 * @return the start run event
	 */
	public String getStartRunEvent() {
		return parsePString("EVNT", 1);
	}

	/**
	 * Get the stop collection event.
	 * 
	 * ABIF tag: EVNT4
	 * 
	 * @return the stop collection event
	 */
	public String getStopCollectionEvent() {
		return parsePString("EVNT", 4);
	}

	/**
	 * Get the stop point.
	 * 
	 * ABIF tag: AEPt2
	 * 
	 * @return the stop point
	 */
	public Short getStopPoint() {
		return parseInteger("AEPt", 2).shortValue();
	}

	/**
	 * Get the stop point (orig).
	 * 
	 * ABIF tag: AEPt1
	 * 
	 * @return the stop point (orig)
	 */
	public Short getStopPointOrig() {
		return parseInteger("AEPt", 1).shortValue();
	}

	/**
	 * Get the stop run event.
	 * 
	 * ABIF tag: EVNT2
	 * 
	 * @return the stop run event
	 */
	public String getStopRunEvent() {
		return parsePString("EVNT", 2);
	}

	/**
	 * Get the temperature, measured in &#176;C.
	 * 
	 * ABIF tag: DATA8
	 * 
	 * @return the temperature, measured in &#176;C
	 */
	public Short[] getTemperature() {
		return charArrayAsShorts(parseAsciiArray("DATA", 8));
	}

	/**
	 * Get the trim probability threshold used.
	 * 
	 * ABIF tag: phTR2
	 * 
	 * @return the trim probability threshold used
	 */
	public Float getTrimProbabilityThreshold() {
		return parseFloat("phTR", 2);
	}

	/**
	 * Get the read positions.
	 * 
	 * ABIF tag: phTR1
	 * 
	 * @return the read positions
	 */
	public Short getTrimRegion() {
		return parseInteger("phTR", 1).shortValue();
	}

	/**
	 * Get the voltage, measured in decavolts.
	 * 
	 * ABIF tag: DATA5
	 * 
	 * @return the voltage, measured in decavolts
	 */
	public Short[] getVoltage() {
		return charArrayAsShorts(parseAsciiArray("DATA", 5));
	}

	/**
	 * Get the name of the user who created the plate.
	 * 
	 * ABIF tag: User1
	 * 
	 * @return the name of the user who created the plate
	 */
	public String getUser() {
		return parsePString("User", 1);
	}

	/**
	 * Get the well ID.
	 * 
	 * ABIF tag: TUBE1
	 * 
	 * @return the well ID
	 */
	public String getWellId() {
		return parsePString("TUBE", 1);
	}

	/**
	 * When using this method it is assumed that we will be pulling 2 byte
	 * shorts out of the data record. This implies that
	 * <code>t.elementLength</code> is equal to 2.
	 * 
	 * @param t
	 *            the data record to treat as an array of shorts
	 * @return the array of shorts from within the record
	 */
	public Short[] parseShortArray(TaggedDataRecord t) {
		Short[] shorts = new Short[(int) t.numberOfElements];

		for (int i = 0, j = 0; i < t.numberOfElements * 2; i += 2, j++) {
			shorts[j] = (short) (t.offsetData[i] << 8 | (t.offsetData[i + 1] & 0xff));
			// shift the first element 8 bits to the left to make it the most
			// significant bit value and then or it with the second element that
			// has been casted into a 2 byte value (& 0xff)
		}

		return shorts;
	}

	/**
	 * A convenience method so that we can just pass the tag name and number
	 * instead of retrieving the record ourself.
	 * 
	 * @see ExtendedABIFParser#parseShortArray(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the short array contained in the data record
	 */
	public Short[] parseShortArray(String tagName, Integer tagNum) {
		return parseShortArray(getDataRecord(tagName, tagNum));
	}

	/**
	 * In most situations you will very likely want to dump this into a string:
	 * 
	 * <code>String s = new String(parseASCIIArray(t));</code>
	 * 
	 * Unfortunately, there will be some situations where this array <b>does
	 * not</b> contain ASCII data but rather is meant to be an array of byte
	 * sized values. Some specific examples of this include CpEP1, PCON1, and
	 * PCON2. CpEP1 is meant to be considered as a single boolean value and
	 * PCON1 and PCON2 are meant to be considered as an array of integer
	 * basecall values. The following is how the ABI document describes these
	 * fields:
	 * 
	 * <pre>
	 *  NAME	NUMBER	DESCRIPTION
	 *  PCON	1	Per-base quality values (edited)
	 *  PCON	2	Per-base quality values
	 *  CpEP	1	Is Capillary Machine?
	 * </pre>
	 * 
	 * It is left to the user of this method to make the distinction between the
	 * two possible types of situations. Fortunately, based upon the tag name
	 * and the ABI documentation you should be able to easily make the
	 * distinction.
	 * 
	 * Note: I would have preferred to use <code>Character[]</code> here, but
	 * apparently there is no <code>String</code> constructor for
	 * <code>Character[]</code>, only <code>char[]</code>.
	 * 
	 * @param t
	 *            the data record to (possibly) treat as an ascii array
	 * @return an ascii array representation of whatever happens to be in the
	 *         record
	 */
	public static char[] parseAsciiArray(TaggedDataRecord t) {
		char[] dataRecord = new char[(int) t.numberOfElements];

		if (t.hasOffsetData) {
			for (int i = 0; i < t.numberOfElements; i++) {
				dataRecord[i] = (char) t.offsetData[i];
			}
		} else {
			for (int i = 0; i < t.numberOfElements; i++) {
				dataRecord[i] = (char) getByteAt(t.dataRecord, i).intValue();
			}
		}

		return dataRecord;
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseAsciiArray(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return an ascii array representation of whatever happens to be in the
	 *         record
	 */
	public char[] parseAsciiArray(String tagName, Integer tagNum) {
		return parseAsciiArray(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>Date</code> from a <code>TaggedDataRecord</code>. Data is
	 * packed into a "date" field as follows:
	 * 
	 * <pre>
	 * struct {
	 * 		SInt16	year;  	// 4-digit year
	 * 		UInt8	month;	// month 1-12
	 * 		UInt8	day;	// day 1-31
	 * }
	 * </pre>
	 * 
	 * @param t
	 *            the record we should treat as a date
	 * @return the date stored within the data record
	 */
	public static Date parseDate(TaggedDataRecord t) {
		Integer year = Long.valueOf(t.dataRecord >>> 16).intValue();
		// year is 2 bytes, so get them both at the same time
		Integer month = getByteAt(t.dataRecord, 2).intValue() - 1;
		// month is 0 based, and thus must have 1 subtracted from it
		Integer day = getByteAt(t.dataRecord, 3).intValue();

		Calendar c = Calendar.getInstance();
		c.set(year, month, day, 0, 0, 0);
		return c.getTime();
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseDate(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the date stored within the data record
	 */
	public Date parseDate(String tagName, Integer tagNum) {
		return parseDate(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>Float</code> from a <code>TaggedDataRecord</code>. Data is
	 * packed into a "float" field as a 4-byte floating point value as per the
	 * ABI spec. Since the <code>long</code> in the dataRecord is already 4
	 * bytes long we just have to cast it into a float.
	 * 
	 * @param t
	 *            the record we should treat as a float
	 * @return the float stored within the data record
	 */
	public static Float parseFloat(TaggedDataRecord t) {
		return Float.intBitsToFloat((int) t.dataRecord);
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseFloat(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the float stored within the data record
	 */
	public Float parseFloat(String tagName, Integer tagNum) {
		return parseFloat(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses an <code>Integer</code> from a <code>TaggedDataRecord</code>. Data
	 * is packed into an "integer" field as a "short" as per the ABI spec. These
	 * values are 16-bit signed integers that are stored in the most significant
	 * bits and thus need to be shifted 16 bits to the right in order to reach
	 * the relevant data. For example, the value stored in a data record is:
	 * 0x00030000. The actual integer value stored in this field is 3. We can
	 * get this value by shifting 16 bits to the right (lop off the least
	 * significant 2 bytes) and then casting the remaining long value into an
	 * int.
	 * 
	 * @param t
	 *            the record we should treat as an integer
	 * @return the integer stored within the record
	 */
	public static Integer parseInteger(TaggedDataRecord t) {
		return Long.valueOf(t.dataRecord >>> 16).intValue();
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseInteger(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the integer stored within the data record
	 */
	public Integer parseInteger(String tagName, Integer tagNum) {
		return parseInteger(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>String</code> from a <code>TaggedDataRecord</code>. Data
	 * is packed into a "pstring" field in the following way. If the whole
	 * string is small enough to be packed in 3 bytes it's stored in the
	 * dataRecord itself. If the string has offset data (ie: it's larger than 3
	 * bytes) then it's stored in offset data and needs to be collected from
	 * that.
	 * 
	 * @param t
	 *            the record we should treat as a string
	 * @return the string stored within the record
	 */
	public static String parsePString(TaggedDataRecord t) {
		StringBuffer dataRecord = new StringBuffer();

		if (t.hasOffsetData) {
			Integer charCount = Long.valueOf(t.numberOfElements).intValue();

			for (Integer i = 1; i < charCount; i++) {
				dataRecord.append((char) t.offsetData[i]);
			}
		} else {
			Integer charCount = Long.valueOf(t.dataRecord >>> 24).intValue();

			for (int i = 1; i <= charCount; i++) {
				dataRecord.append((char) getByteAt(t.dataRecord, i).intValue());
			}
		}

		return dataRecord.toString();
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parsePString(TaggedDataRecord t)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the string stored within the data record
	 */
	public String parsePString(String tagName, Integer tagNum) {
		return parsePString(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>String</code> from a <code>TaggedDataRecord</code>. Data
	 * is packed into a "date" field as follows:
	 * 
	 * <pre>
	 * struct {
	 * 		UInt8	hour;  	// hour 0-23
	 * 		UInt8	minute;	// minute 0-59
	 * 		UInt8	second;	// second 0-59
	 * 		UInt8	hsec;	// 0.01 second 0-99
	 * }
	 * </pre>
	 * 
	 * We pull the time from the record in a format such that it is readily
	 * parsed by <code>SimpleDateFormat</code>. The values are formatted as
	 * such:
	 * 
	 * HH:mm:ss.SS
	 * 
	 * @param t
	 *            the record we should treat as a time
	 * @return the time stored within the data record
	 */
	public static String parseTime(TaggedDataRecord t) {
		NumberFormat twoDigits = new DecimalFormat("00");

		Integer hour = getByteAt(t.dataRecord, 0).intValue();
		Integer minute = getByteAt(t.dataRecord, 1).intValue();
		Integer second = getByteAt(t.dataRecord, 2).intValue();
		Integer hsecond = getByteAt(t.dataRecord, 3).intValue();

		return twoDigits.format(hour) + ":" + twoDigits.format(minute) + ":"
				+ twoDigits.format(second) + "." + twoDigits.format(hsecond);

	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseTime(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the time stored within the data record
	 */
	public String parseTime(String tagName, Integer tagNum) {
		return parseTime(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>String</code> from a <code>TaggedDataRecord</code>. Data
	 * is packed into a "pstring" field in the following way. If the whole
	 * string is small enough to be packed in 3 bytes it's stored in the
	 * dataRecord itself. If the string has offset data (ie: it's larger than 3
	 * bytes) then it's stored in offset data and needs to be collected from
	 * that.
	 * 
	 * @param t
	 *            the record we should treat as a string
	 * @return the string stored within the record
	 */
	public static String parseCString(TaggedDataRecord t) {
		StringBuffer dataRecord = new StringBuffer();

		if (t.hasOffsetData) {
			for (int i = 0; i < t.numberOfElements; i++) {
				if (t.offsetData[i] != 0) {
					dataRecord.append((char) t.offsetData[i]);
				}
			}
		} else {
			for (int i = 0; i < 3; i++) {
				byte currentByte = getByteAt(t.dataRecord, i);
				if (currentByte != 0) {
					dataRecord.append((char) currentByte);
				} else {
					break;
				}
			}
		}

		return dataRecord.toString();
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseCString(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the string stored within the data record
	 */
	public String parseCString(String tagName, Integer tagNum) {
		return parseCString(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>Long</code> from a <code>TaggedDataRecord</code>. Data is
	 * packed into a "long" field as a 32 bit integer value according to the ABI
	 * spec. Thus, we simply return the data record as is, no shifting or
	 * manipulation is otherwise required.
	 * 
	 * @param t
	 *            the record we should treat as a long
	 * @return the long stored within the record
	 */
	public static Long parseLong(TaggedDataRecord t) {
		return t.dataRecord;
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseLong(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the long stored within the data record
	 */
	public Long parseLong(String tagName, Integer tagNum) {
		return parseLong(getDataRecord(tagName, tagNum));
	}

	/**
	 * Parses a <code>Byte</code> from a <code>TaggedDataRecord</code>. Data is
	 * packed into a "byte" field as a single byte as per the ABI spec. This
	 * value is stored in the 8 most significant bits within the data record and
	 * thus the data record needs to be shifted to the right 24 bits.
	 * 
	 * @param t
	 *            the record we should treat as a byte
	 * @return the byte stored within the record
	 */
	public static Byte parseByte(TaggedDataRecord t) {
		return Long.valueOf(t.dataRecord >>> 24).byteValue();
	}

	/**
	 * A convenience method so we can just pass a tag name and number instead of
	 * having to retrieve the data record ourself.
	 * 
	 * @see ExtendedABIFParser#parseByte(TaggedDataRecord)
	 * @param tagName
	 *            the name of the tag
	 * @param tagNum
	 *            the number of the tag
	 * @return the byte stored within the data record
	 */
	public Byte parseByte(String tagName, Integer tagNum) {
		return parseByte(getDataRecord(tagName, tagNum));
	}

	private static Byte getByteAt(Long record, int position) {
		return Long
				.valueOf(
						(record & (0xFF000000 >>> (position * 8))) >>> ((3 - position) * 8))
				.byteValue();
		// YEGADS! Okay, I'll explain:
		// As the value of i increases we will successively isolate the
		// current byte that we want to examine. The next line will
		// shift that byte to the appropriate position (ie, the least
		// significant bits) so that we can cast it into a char and
		// append to our string
	}

	private Short[] charArrayAsShorts(char[] chars) {
		Short[] shorts = new Short[chars.length];

		for (int i = 0; i < chars.length; i++) {
			shorts[i] = (short) chars[i];
		}

		return shorts;
	}

	private Byte[] charArrayAsBytes(char[] chars) {
		Byte[] bytes = new Byte[chars.length];

		for (int i = 0; i < chars.length; i++) {
			bytes[i] = (byte) chars[i];
		}

		return bytes;
	}

	public Long[] charArrayAsLongs(char[] chars) {
		Long[] longs = new Long[chars.length / 4];

		for (int i = 0; i < chars.length; i += 4) {
			longs[i] = (long) (chars[i] << 24 | chars[i + 1] << 16
					| chars[i + 2] << 8 | chars[i + 3]);
		}

		return longs;
	}

	/**
	 * Add a the specified time to the date using the specified time format to
	 * parse the time.
	 * 
	 * @param date
	 *            the date we should be adding to
	 * @param time
	 *            the time we should add to the date
	 * @param timeFormat
	 *            the way we should be parsing the specified time
	 * @return the date with the time added to it
	 */
	private Date addTimeToDate(Date date, String time, String timeFormat) {
		DateFormat formatter = new SimpleDateFormat(timeFormat);
		Calendar operatingDate = Calendar.getInstance();
		Calendar operatingTime = Calendar.getInstance();
		try {
			operatingDate.setTime(date);
			operatingTime.setTime(formatter.parse(time));

			operatingDate.set(Calendar.HOUR, operatingTime.get(Calendar.HOUR));
			operatingDate.set(Calendar.MINUTE, operatingTime
					.get(Calendar.MINUTE));
			operatingDate.set(Calendar.SECOND, operatingTime
					.get(Calendar.SECOND));
			operatingDate.set(Calendar.MILLISECOND, operatingTime
					.get(Calendar.MILLISECOND));
			operatingDate
					.set(Calendar.AM_PM, operatingTime.get(Calendar.AM_PM));
		} catch (ParseException e) {
			e.printStackTrace();
		}

		return operatingDate.getTime();
	}

	/**
	 * An extended version of {@link TaggedDataRecord} that contains all of the
	 * possible data types and read methods that are described in the ABI
	 * documentation. As a side note, crypticVariable in super is actually
	 * referred to as datahandle in the ABI spec. This is described as:
	 * 
	 * <blockquote> Your implementation should ignore the datahandle field on
	 * input and write a value of zero on output. (This field was used in
	 * libraries that implemented a virtual memory system similar to that of the
	 * Classic Mac OS Resource manager; see "Historical Notes" on page 5.)
	 * </blockquote>
	 * 
	 * @author Franklin Bristow (franklin_bristow@phac-aspc.gc.ca)
	 * 
	 */
	public static class ExtendedTaggedDataRecord extends TaggedDataRecord {
		/* Current Data Types */
		public static final int DATA_TYPE_BYTE = 1;
		public static final int DATA_TYPE_WORD = 3;
		public static final int DATA_TYPE_LONG = 5;
		public static final int DATA_TYPE_DOUBLE = 8;
		public static final int DATA_TYPE_CSTRING = 19;

		/* Supported Legacy Types */
		public static final int DATA_TYPE_THUMB = 12;
		public static final int DATA_TYPE_BOOL = 13;

		/* Unsupported Legacy Types */
		@Deprecated
		public static final int DATA_TYPE_RATIONAL = 6;
		@Deprecated
		public static final int DATA_TYPE_BCD = 9;
		@Deprecated
		public static final int DATA_TYPE_POINT = 14;
		@Deprecated
		public static final int DATA_TYPE_RECT = 15;
		@Deprecated
		public static final int DATA_TYPE_VPOINT = 16;
		@Deprecated
		public static final int DATA_TYPE_VRECT = 17;
		@Deprecated
		public static final int DATA_TYPE_TAG = 20;
		@Deprecated
		public static final int DATA_TYPE_DELTA_COMP = 128;
		@Deprecated
		public static final int DATA_TYPE_LZW_COMP = 256;
		@Deprecated
		public static final int DATA_TYPE_DELTA_LZW = 384;

		public ExtendedTaggedDataRecord(DataAccess dataAccess)
				throws IOException {
			super(dataAccess);
		}
	}
}
